/*
Notice: This computer software was prepared by Brookhaven Science Associates, LLC and [individual author],
hereinafter the Contractor, under Contract DE-AC02-98CH10886 with the Department of Energy (DOE).
All rights in the computer software are reserved by DOE on behalf of the United States Government and the Contractor as
provided in the Contract. You are authorized to use this computer software for
Governmental purposes but it is not to be released or distributed to the public. NEITHER
THE GOVERNMENT NOR THE CONTRACTOR MAKES ANY WARRANTY,
EXPRESS OR IMPLIED, OR ASSUMES ANY LIABILITY FOR THE USE OF THIS
SOFTWARE. This notice including this sentence must appear on any copies of this
computer software.
(End of Notice)
*/
#include <string.h>
#include <ctype.h>
#include <XmlDom/XDDocument.hxx>
#include <XmlDom/XDNode.hxx>
#include <xercesc/util/XMLString.hpp>
#include <xercesc/util/PlatformUtils.hpp>
#include <xercesc/dom/DOMAttr.hpp>
#include <xercesc/dom/DOMNamedNodeMap.hpp>
#include <xercesc/dom/DOMException.hpp>
#include <xercesc/dom/DOMDocument.hpp>
#include <xercesc/parsers/XercesDOMParser.hpp>

//////////////////////////////////////////////////////////////
int XDDocument::_numObjects = 0;

XDDocument::XDDocument()
{
  initialize();
  _numObjects++;
  if(_numObjects == 1)
  {
    try {
      XMLPlatformUtils::Initialize();
    }
    catch(const XMLException& err) {
      fprintf(stderr, "Error during Xerces Initialization.\n");
    }
  }
}


XDDocument::~XDDocument()
{
  clear();
  _numObjects--;
  if(_numObjects == 0)
      XMLPlatformUtils::Terminate();
}

void XDDocument::initialize()
{
  _filename = _dtdFilename = _dtdName = NULL;
  _rootNode = NULL;
  _lastError[0] = _lastComment[0] = 0;
  strcpy(_version, "1.0");
  strcpy(_encoding, "UTF-8");
  strcpy(_standalone, "yes");
}

void XDDocument::clear()
{
  free(_filename);
  free(_dtdFilename);
  free(_dtdName);
  
  // delete any existing tree
  deleteNode(_rootNode);
  
  initialize();
}

void XDDocument::deleteNode(XDNode* node)
{
  // this routine recursively deletes the node and any children
  if(node == NULL)
      return;
  
  // first delete children nodes
  XDNode* curNode = (XDNode*) node->getFirstChild();
  XDNode* nextNode;
  while(curNode != NULL)
  {
    nextNode = (XDNode*) curNode->getNextSibling();
    deleteNode(curNode);
    curNode = nextNode;
  }
  // now delete the node
  delete node;
}

int XDDocument::read(const char* filename, bool validate)
{
  // create an XML parser and parse the file
  //  not sure whether we need all of this setup stuff
  XercesDOMParser parser;
  if(validate)
    parser.setValidationScheme(XercesDOMParser::Val_Auto);
  else
    parser.setValidationScheme(XercesDOMParser::Val_Never);
  parser.setDoNamespaces(false);
  parser.setDoSchema(false);
  parser.setValidationSchemaFullChecking(false);
  parser.setErrorHandler(this);
  //
  // XXX: What does this do!?
  //parser.setToCreateXMLDeclTypeNode(true);
  _lastError[0] = 0;
  try {
    parser.parse(filename);
  }
  catch (const XMLException& e) {
    if(_lastError[0] == 0)
      sprintf(_lastError, "XML exception occurred while reading file.");
    return -1;
  }
  catch (const DOMException& e) {
    if(_lastError[0] == 0)
      sprintf(_lastError, "XML DOM exception occurred while reading file.");
    return -2;
  }
  catch (...) {
    if(_lastError[0] == 0)
      sprintf(_lastError, "Generic exception occurred while reading file.");
    return -3;
  }
  if(_lastError[0] != 0)  // error caught by error handling callbacks
    return -4;
  
  // is this an XML file? store XML info
  DOMDocument* doc = parser.getDocument();
  if(doc == NULL) {
    sprintf(_lastError, "Document is NULL.");
    return -3;
  }
  DOMNode* firstNode = doc->getFirstChild();
  if(firstNode == NULL) {
    sprintf(_lastError, "firstNode is NULL.");
    return -3;
  }
#ifndef __CYGWIN // CYGWIN 1.7 uses a laetr version of xerces
#ifndef __WS5
  char* version = XMLString::transcode(doc->getVersion()); 
  if(version && version[0])
    {
      strcpy(_version, version);
      XMLString::release(&version);
    }
  char* encoding = XMLString::transcode(doc->getEncoding()); 
  if(encoding && encoding[0])
    {
      strcpy(_encoding, encoding);
      XMLString::release(&encoding);
    }
  bool standalone = doc->getStandalone();
  if(standalone){
    strcpy(_standalone, "yes");
  }
#endif
#endif //CYGWIN
  // prepare for creation of a new document tree
  clear();
  setFilename(filename);
  
  // store the DTD name and filename, if any
  DOMDocumentType* docTypeP = doc->getDoctype();
  if(docTypeP != NULL)
    {
      char* name = XMLString::transcode(docTypeP->getName()); 
      if(name && name[0] != 0){
        setDtdName(name);
	XMLString::release(&name);
      }
      char* sysId = XMLString::transcode(docTypeP->getSystemId());
      if(sysId && sysId[0] != 0){
        setDtdFilename(sysId);
	XMLString::release(&sysId);
      }
    }
  
  // the next node better indicate XML and encoding type
  // load the DOM tree into our local tree as appropriate
  // by recursively loading the nodes
  loadNode(doc, NULL);
  return 0;
}

int XDDocument::loadNode(DOMNode* dnode, XDNode* parent)
{
  char* name = XMLString::transcode(dnode->getNodeName()); 
  char* value = XMLString::transcode(dnode->getNodeValue()); 
  
  switch( dnode->getNodeType() )
  {
  case DOMNode::TEXT_NODE:
  case DOMNode::CDATA_SECTION_NODE:
    {
      // this is the text associated with the parent
      value = stripWhitespace(value);  // get rid of \n only nodes and leading/trailing whitespace
      if(value && value[0] != 0)
        {
          if(dnode->getNodeType() == DOMNode::CDATA_SECTION_NODE)
            parent->setValueWrapped(value);
          else
            parent->setValue(value);
        }
    }
    break;
  case DOMNode::ELEMENT_NODE :
    {
      // create the new node
      XDNode* node;
      if(parent != NULL)
          node = parent->appendChild(name, value);
      else
      {
        node = new XDNode(name, value);
        _rootNode = node;
      }
      parent = node;

      // add the attributes
      DOMNamedNodeMap& attributes = *dnode->getAttributes();
      int attrCount = attributes.getLength();
      XDAttr* attr = NULL;
      char *aname = NULL, *avalue = NULL;
      for(int i = 0; i < attrCount; i++)
      {
        DOMAttr& dattr = (DOMAttr&) *attributes.item(i);
        aname = XMLString::transcode(dattr.getNodeName()); 
        avalue = XMLString::transcode(dattr.getNodeValue()); 
        attr = node->addAttr(aname, avalue);
        XMLString::release(&aname);
	XMLString::release(&avalue);
        attr->setSpecified( dattr.getSpecified() );
      }
      // if there is a comment for this element, store it
      if(_lastComment[0] != 0)
        node->setComment(_lastComment);
      _lastComment[0] = 0;
    }
    break;
  case DOMNode::COMMENT_NODE:
    // cache this information to be attached to next element
    if(strlen(_lastComment) + strlen(value) < 1024)
        strcat(_lastComment, value);
    break;
  case DOMNode::DOCUMENT_NODE :
    break;
  }
  XMLString::release(&name);
  XMLString::release(&value);

  // create nodes (recursively) for children if necessary
  DOMNode* cnode = dnode->getFirstChild();
  while(cnode != NULL)
    {
      loadNode(cnode, parent);
      cnode = cnode->getNextSibling();
    }
  return 0;
}

char* XDDocument::stripWhitespace(char* str)
{
  // replaces any trailing whitespac characters with the NULL character
  if(str == NULL)
    return NULL;
  int i, len = strlen(str);

  // if there is leading space, create a new string and return it
  if( isspace(str[0]) )
    {
      char* newstr = (char*) malloc(len+1);
      for(i=0; i<len; i++)
        {
          if( !isspace(str[i]) )
            break;
        }
      if(i == len)  // all whitespace
        {
          free(newstr);
          free(str);
          return NULL;
        }
      strcpy(newstr, &str[i]);
      free(str);
      return stripWhitespace(newstr);  // now possibly just trailing whitespace
    }
  // if no leading spaces, just lop off the whitespace at the end
  else
    {
      for(i=len-1; i>=0; i--)
        {
          if( isspace(str[i]) )
            str[i] = 0;
          else
            break;
        }
      if( i < 0)  // all whitespace
        return NULL;
      else
        return str;
    }
}

void XDDocument::warning(const SAXParseException&)
{
  // Ignore all warnings.
}

void XDDocument::error(const SAXParseException& err)
{
  char* errstr = XMLString::transcode(err.getMessage());
  int line = err.getLineNumber();
  int col = err.getColumnNumber();
  if(errstr == NULL) {
    snprintf(_lastError, 1024, "Error at line %d, column %d", line, col);
  }
  else {
    snprintf(_lastError, 1024, "Error at line %d, column %d - %s",
             line, col, errstr);
    XMLString::release(&errstr);
  }
}

void XDDocument::fatalError(const SAXParseException& err)
{
  char* errstr = XMLString::transcode(err.getMessage()); 
  int line = err.getLineNumber();
  int col = err.getColumnNumber();
  if(errstr == NULL) {
    snprintf(_lastError, 1024, "Fatal Error at line %d, column %d", line, col);
  }
  else {
    snprintf(_lastError, 1024, "Fatal Error at line %d, column %d - %s",
             line, col, errstr);
    XMLString::release(&errstr);
  }
}

void XDDocument::resetErrors()
{
  _lastError[0] = 0;
}

const char* XDDocument::getLastError() const
{
  return _lastError;
}

int XDDocument::write(const char* filename, bool validate)
{
  _lastError[0] = 0;
  if(filename == NULL)
    {
      sprintf(_lastError, "NULL filename passed to write()");
      return -1;
    }
  if( _rootNode == NULL)
    {
      sprintf(_lastError, "No root node for document - nothing to write.");
      return -1;
    }
  
  // open the file
  FILE* fp = fopen(filename, "w");
  if(fp == NULL)
    {
      sprintf(_lastError, "Could not open file %s.", filename);
      return -1;
    }

  // write to the open file ptr
  int retval = write(fp);
  
  // close the file
  fclose(fp);

  // if validate=true and there is a DTD file, reread the file
  // with validation to see if it is valid
  if(validate && _dtdFilename)
    retval = read(filename, true);

  return retval;
}

int XDDocument::write(FILE* fp)
{
  if(fp == NULL)
    return -1;

  // write out the header area
  fprintf(fp, "<?xml version=\"%s\" encoding=\"%s\" standalone=\"%s\"?>\n", _version, _encoding, _standalone);
  if(_dtdFilename && _dtdFilename[0] != 0)
    {
      if(_dtdName && _dtdName[0] != 0)
        fprintf(fp, "<!DOCTYPE %s SYSTEM \"%s\">\n", _dtdName, _dtdFilename);
      else if(_rootNode && _rootNode->getName() )
        fprintf(fp, "<!DOCTYPE %s SYSTEM \"%s\">\n", _rootNode->getName(), _dtdFilename);
    }

  // write out the tree recursively
  writeNode(fp, _rootNode, 0);
  return 0;
}

void XDDocument::writeNode(FILE* fp, const XDNode* node, short indentLevel)
{
  // this routine recursively writes a single node out to fp
  if(node == NULL)
    return;
  const char* name = node->getName();
  const char* value = node->getValue();
  const char* comment = node->getComment();
  
  // write out any comment associated with this node
  char lineStr[1024], tmpStr[256];
  if(comment != NULL)
    {
      lineStr[0] = 0;
      for(short i=0; i<indentLevel; i++)  // indent
	strcat(lineStr, "  ");
      sprintf(tmpStr, "<!--%s-->\n", comment);
      strcat(lineStr, tmpStr);
      fputs(lineStr, fp);
    }
  
  // write the start tag
  lineStr[0] = 0;
  for(short i=0; i<indentLevel; i++)  // indent
    strcat(lineStr, "  ");
  sprintf(tmpStr, "<%s", name);
  strcat(lineStr, tmpStr);
  
  // write the attributes
  const XDAttr* attr = node->getFirstAttr();
  while(attr != NULL)
  {
    sprintf(tmpStr, " %s=\"%s\"", attr->getName(), attr->getValue() );
    strcat(lineStr, tmpStr);
    attr = attr->getNextAttr();
  }
  strcat(lineStr, ">");
  
  // write the text
  if(value != NULL)
    {
      if(node->isValueWrapped() == true)
        strcat(lineStr, "<![CDATA[");
      strcat(lineStr, value);
      if(node->isValueWrapped() == true)
        strcat(lineStr, "]]>");
    }
  // handle children
  const XDNode* childNode = node->getFirstChild();
  if(childNode != NULL)
  {
    // write out existing text now and start new line
    strcat(lineStr, "\n");
    fputs(lineStr, fp);
    
    // recursively write out the children nodes
    while(childNode!= NULL)
    {
      writeNode(fp, childNode, indentLevel+1);
      childNode = childNode->getNextSibling();
    }
    // write out the (indented) end tag line
    lineStr[0] = 0;
    for(short i=0; i<indentLevel; i++)
      strcat(lineStr, "  ");
    sprintf(tmpStr, "</%s>\n", name);
    strcat(lineStr, tmpStr);
    fputs(lineStr, fp);
  }
  else  // no children
  {
    // append the end tag and write the line
    sprintf(tmpStr, "</%s>\n", name);
    strcat(lineStr, tmpStr);
    fputs(lineStr, fp);
  }
}

void XDDocument::setFilename(const char* filename)
{
  free(_filename);
  _filename = NULL;
  if(filename)
      _filename = strdup(filename);
}

const char* XDDocument::getFilename() const
{
  return _filename;
}

void XDDocument::setDtdFilename(const char* filename)
{
  free(_dtdFilename);
  _dtdFilename = NULL;
  if(filename)
    {
      _dtdFilename = strdup(filename);
      strcpy(_standalone, "no");
    }
}

const char* XDDocument::getDtdFilename() const
{
  return _dtdFilename;
}

void XDDocument::setDtdName(const char* name)
{
  free(_dtdName);
  _dtdName = NULL;
  if(name)
    _dtdName = strdup(name);
}

const char* XDDocument::getDtdName() const
{
  return _dtdName;
}

const XDNode* XDDocument::getRootNode() const
{
  return _rootNode;
}

void XDDocument::setRootNode(XDNode* root)
{
  // first delete out old, if the user hasn't already
  if(_rootNode != NULL)
      clear();
  
  _rootNode = root;
}

const char* XDDocument::getVersion() const
{
  return _version;
}

void XDDocument::setVersion(const char* version)
{
  strcpy(_version, version);
}

const char* XDDocument::getEncoding() const
{
  return _encoding;
}

void XDDocument::setEncoding(const char* encoding)
{
  strcpy(_encoding, encoding);
}

const char* XDDocument::getStandalone() const
{
  return _standalone;
}

void XDDocument::setStandalone(const char* standalone)
{
  strcpy(_standalone, standalone);
}

