/*
Notice: This computer software was prepared by Brookhaven Science Associates, LLC and [individual author],
hereinafter the Contractor, under Contract DE-AC02-98CH10886 with the Department of Energy (DOE).
All rights in the computer software are reserved by DOE on behalf of the United States Government and the Contractor as
provided in the Contract. You are authorized to use this computer software for
Governmental purposes but it is not to be released or distributed to the public. NEITHER
THE GOVERNMENT NOR THE CONTRACTOR MAKES ANY WARRANTY,
EXPRESS OR IMPLIED, OR ASSUMES ANY LIABILITY FOR THE USE OF THIS
SOFTWARE. This notice including this sentence must appear on any copies of this
computer software.
(End of Notice)
*/
#include <string.h>
#include <stdio.h>
#include <XmlDom/XDNode.hxx>

//////////////////////////////////////////////////////////////
XDNode::XDNode(const char* name, const char* value)
{
  initialize();
  if(name)
    setName(name);
  if(value)
    setValue(value);
}

XDNode::~XDNode()
{
  free(_name);
  free(_value);
  free(_comment);

  // delete all of the node attributes
  XDAttr* attr = _firstAttr;
  XDAttr* nextAttr;
  while(attr != NULL)
    {
      nextAttr = (XDAttr*) attr->getNextAttr();
      delete attr;
      attr = nextAttr;
    }
}

void XDNode::initialize()
{
  _name = _value = _comment = NULL;
  _parent = _firstChild = _nextSibling = NULL;
  _firstAttr = NULL;
  _wrapValue = false;
}

const char* XDNode::getName() const
{
  return _name;
}

const char* XDNode::getValue() const
{
  return _value;
}

const char* XDNode::getComment() const
{
  return _comment;
}

void XDNode::setName(const char* name)
{
  free(_name);
  _name = NULL;
  if(name)
    _name = strdup(name);
}

void XDNode::setValue(const char* value)
{
  free(_value);
  _value = NULL;
  if(value)
    _value = strdup(value);
  _wrapValue = false;
}

void XDNode::setValue(double value)
{
  char tmp[64];
  snprintf(tmp, 64, "%g", value);
  setValue((const char*)tmp);
}

void XDNode::setValue(float value)
{
  char tmp[64];
  snprintf(tmp, 64, "%f", value);
  setValue((const char*)tmp);
}

void XDNode::setValue(int value)
{
  char tmp[64];
  snprintf(tmp, 64, "%d", value);
  setValue((const char*)tmp);
}

void XDNode::setValueWrapped(const char* value)
{
  setValue(value);
  _wrapValue = true;
}

bool XDNode::isValueWrapped() const
{
  return _wrapValue;
}

void XDNode::setComment(const char* comment)
{
  free(_comment);
  _comment = NULL;
  if(comment)
    _comment = strdup(comment);
}

const XDNode* XDNode::getParent() const
{
  return _parent;
}

const XDNode* XDNode::getFirstChild() const
{
  return _firstChild;
}

const XDNode* XDNode::getNextSibling() const
{
  return _nextSibling;
}

XDNode* XDNode::appendChild(const char* name, const char* value)
{
  // create a new node
  XDNode* node = new XDNode(name, value);

  // append it onto the end of the children list
  XDNode* lastChild = getLastChild();
  if(lastChild == NULL)
    _firstChild = node;
  else
    lastChild->_nextSibling = node;
  return node;
}

XDNode* XDNode::appendSibling(const char* name, const char* value)
{
  // create a new node and append it
  XDNode* node = new XDNode(name, value);
  _nextSibling = node;
  return node;
}

XDNode* XDNode::appendChildWrapped(const char* name, const char* value)
{
  // create a new node with a wrapped value
  XDNode* node = new XDNode(name);
  node->setValueWrapped(value);

  // append it onto the end of the children list
  XDNode* lastChild = getLastChild();
  if(lastChild == NULL)
    _firstChild = node;
  else
    lastChild->_nextSibling = node;
  return node;
}

XDNode* XDNode::appendSiblingWrapped(const char* name, const char* value)
{
  // create a new node and append it
  XDNode* node = new XDNode(name);
  node->setValueWrapped(value);
  _nextSibling = node;
  return node;
}

XDNode* XDNode::getLastChild() const
{
  if(_firstChild == NULL)
    return NULL;
  const XDNode* child = _firstChild;
  while(child->getNextSibling() != NULL)
    child = child->getNextSibling();
  return (XDNode*) child;
}

const XDAttr* XDNode::getFirstAttr() const
{
  return _firstAttr;
}

const XDAttr* XDNode::getAttr(const char* name) const
{
  const XDAttr* attr = _firstAttr;
  while(attr != NULL)
    {
      if( !strcmp(attr->getName(), name) )
        return attr;
      attr = attr->getNextAttr();
    }
  return NULL;
}

XDAttr* XDNode::addAttr(const char* name, const char* value)
{
  // create a new attribute
  XDAttr* attr = new XDAttr(name, value);

  // append it onto the end of the attribute list
  XDAttr* lastAttr = getLastAttr();
  if(lastAttr == NULL)
    _firstAttr = attr;
  else
    lastAttr->_nextAttr = attr;
  return attr;
}

XDAttr* XDNode::getLastAttr() const
{
  if(_firstAttr == NULL)
    return NULL;
  const XDAttr* attr = _firstAttr;
  while(attr->getNextAttr() != NULL)
    attr = attr->getNextAttr();
  return (XDAttr*) attr;
}

const XDNode* XDNode::getChild(const char* name) const
{
  if(!name)
    return NULL;

  const XDNode* child = getFirstChild();
  while(child)
    {
      if(!strcmp(child->getName(), name))
	return child;
      child = child->getNextSibling();
    }

  return NULL;
}

//////////////////////////////////////////////////////////////
XDAttr::XDAttr(const char* name, const char* value)
{
  initialize();
  if(name)
    setName(name);
  if(value)
    setValue(value);
}

XDAttr::~XDAttr()
{
  free(_name);
  free(_value);
}

void XDAttr::initialize()
{
  _name = _value = NULL;
  _specified = false;
  _nextAttr = NULL;
}

const char* XDAttr::getName() const
{
  return _name;
}

const char* XDAttr::getValue() const
{
  return _value;
}

void XDAttr::setName(const char* name)
{
  free(_name);
  _name = NULL;
  if(name)
    _name = strdup(name);
}

void XDAttr::setValue(const char* value)
{
  free(_value);
  _value = NULL;
  if(value)
    _value = strdup(value);
}

void XDAttr::setValue(double value)
{
  char tmp[64];
  snprintf(tmp, 64, "%g", value);
  setValue((const char*)tmp);
}

void XDAttr::setValue(float value)
{
  char tmp[64];
  snprintf(tmp, 64, "%f", value);
  setValue((const char*)tmp);
}

void XDAttr::setValue(int value)
{
  char tmp[64];
  snprintf(tmp, 64, "%d", value);
  setValue((const char*)tmp);
}

bool XDAttr::isSpecified() const
{
  return _specified;
}

void XDAttr::setSpecified(bool specified)
{
  _specified = specified;
}

const XDAttr* XDAttr::getNextAttr() const
{
  return _nextAttr;
}

