/*
Notice: This computer software was prepared by Brookhaven Science Associates, LLC and [individual author],
hereinafter the Contractor, under Contract DE-AC02-98CH10886 with the Department of Energy (DOE).
All rights in the computer software are reserved by DOE on behalf of the United States Government and the Contractor as
provided in the Contract. You are authorized to use this computer software for
Governmental purposes but it is not to be released or distributed to the public. NEITHER
THE GOVERNMENT NOR THE CONTRACTOR MAKES ANY WARRANTY,
EXPRESS OR IMPLIED, OR ASSUMES ANY LIABILITY FOR THE USE OF THIS
SOFTWARE. This notice including this sentence must appear on any copies of this
computer software.
(End of Notice)
*/
#include <stdlib.h>
#include <stdio.h>
#include <sys/times.h>

#define LINUX
#include "CAENVMElib.h"
#undef LINUX

static int mask(int nBits){
  switch(nBits){
  case 1:  return 0x0001;
  case 2:  return 0x0003;
  case 3:  return 0x0007;
  case 4:  return 0x000f;
  case 5:  return 0x001f;
  case 6:  return 0x003f;
  case 7:  return 0x007f;
  case 8:  return 0x00ff;
  case 9:  return 0x01ff;
  case 10: return 0x03ff;
  case 11: return 0x07ff;
  case 12: return 0x0fff;
  case 13: return 0x1fff;
  case 14: return 0x3fff;
  case 15: return 0x7fff;
  case 16: return 0xffff;
  case 32: return 0x0fff0fff; /* special case! */
  default: return 0;
  }
}

static long BHandle = 0;

//static volatile int *addr(unsigned long base, int offset){ return ((volatile int *)(base + (offset<<8))); }

static int readRegister(unsigned long base, int offset, int nBits){

  int val = 0;

  (void) CAENVME_ReadCycle(BHandle,(unsigned long) (base + (offset<<8)), &val,cvA32_U_DATA,cvD32);
  return (mask(nBits) & val);

}

static void writeRegister(unsigned long base, int offset, int nBits, int val){
  
  val &= mask(nBits);

  (void) CAENVME_WriteCycle(BHandle,(unsigned long) (base + (offset<<8)), &val,cvA32_U_DATA,cvD32);

}

static void arm(unsigned long base){

  writeRegister(base, 0x17, 1, ~0);
}

static void reset(unsigned long base) { writeRegister(base, 0x8, 1, ~0); }

static int triggerSource(unsigned long base, int src){

  if(src>=0 && src <= 1<<5) writeRegister(base, 0x1d, 5, src);

  return readRegister(base, 0x1d, 5);

}

static int nSamples(unsigned long base, int nSamples){

   if(nSamples >= 0 && nSamples <= 128*20) writeRegister(base, 0x22, 8, nSamples/20);

   return (20 * readRegister(base, 0x22, 8));
}

static int checkAcq(unsigned long base){

    return(readRegister(base, 0x80, 1));
}

static int postTrigger(unsigned long base, int nSamples){
/* this is really in 20X samples... */

   if(nSamples >= 0 && nSamples <= 0xffff*20) {
       nSamples/=20;
       writeRegister(base, 0x1a, 8, nSamples&0xff);
       writeRegister(base, 0x1b, 8, nSamples>>8);
   }

   return (20*(readRegister(base, 0x1b, 8)<<8 | readRegister(base, 0x1a, 8)));
}

static int frequency(unsigned long base, int freq){
/* 1 = 2GHz, 0 = 1GHz */

        if((freq & ~1) == freq) writeRegister(base, 0x81, 1, freq);

        return readRegister(base, 0x81, 1);

}

#include <time.h>
static void waitAcq(unsigned long base, int timeout){

    const struct timespec pause = {0, 500000};
    int now = time(0);
    /* taskDelay(2); return; */
    while(checkAcq(base)==0 && (time(0)-now) <= timeout) nanosleep(&pause,NULL);
}


static void readRam(unsigned long base, int chan, const short *background, short *buf1, short *buf2, short*buf3, short *buf4){
  
  int nSamp = nSamples(base, -1);

  int postSamples = 2560 + 2520 - (postTrigger(base, -1)%2560);
  
  int i;

  int trigRec = 20 * (/*128 -*/ readRegister(base, 0x20, 8));

  /* start at beginning */
  reset(base);

  /* dump onto floor */
  for(i=0;i<3*4;i+=2){ (void) readRegister(base, 0xd, 32); }

  for(i=0;i<nSamp*4;i+=2){
    int temp = readRegister(base, 0xd, 32);
    int r1 = temp>>16, r2 = temp & 0x0fff;
    if(background) { r1 -= background[i]; r2 -= background[i+1];}
#if(0)
    switch(chan%4){
    case 0: if(i%4!=0) buf[(i/4+trigRec+postSamples)%2560]=r2; break;
    case 1: if(i%4!=0) buf[(i/4+trigRec+postSamples)%2560]=r1; break;
    case 2: if(i%4==0) buf[(i/4+trigRec+postSamples)%2560]=r2; break;
    case 3: if(i%4==0) buf[(i/4+trigRec+postSamples)%2560]=r1; break;
    }
#endif
    if(i%4!=0) buf1[(i/4+trigRec+postSamples)%2560]=r2;
    if(i%4!=0) buf2[(i/4+trigRec+postSamples)%2560]=r1;
    if(i%4==0) buf3[(i/4+trigRec+postSamples)%2560]=r2;
    if(i%4==0) buf4[(i/4+trigRec+postSamples)%2560]=r1;

    if(0){
      double v1 = ((double)(r1-2048))/4.096;
      double v2 = ((double)(r2-2048))/4.096;
      printf("c%c[%i]:%12g\t%s", i%4==0?'3':'1', i/4, v1, i%8==0?"":"");
      printf("c%c[%i]:%12g\t%s", i%4==0?'2':'0', i/4, v2, i%8==0?"\n":"");
    }
  }
  /* printf("\n");         */
}

#define ITER 5
#define NUM_CHANS 4
void takeBackground(unsigned long base, FILE *baseline){
  
  int i, j;

  int nSamp = nSamples(base, -1);

  /* user will need to delete this */
  unsigned long *buf = calloc(sizeof(buf[0]), NUM_CHANS*(nSamp+3));

  for(i=0;i<(1<<ITER);i++){

  reset(base);
  triggerSource(base, 0x8); /* random */
  arm(base);
  waitAcq(base, 1000);

  for(j=0;j<(nSamp+3)*NUM_CHANS;j+=2){
    int temp = readRegister(base, 0xd, 32);
    buf[j] += (temp >> 16);
    buf[j+1] += (temp & 0x0fff);
    }
  }

  for(j=0;j<(nSamp+3)*NUM_CHANS;j++){
    buf[j] = buf[j]>>ITER;
  }

  for(i=0;i<NUM_CHANS;i++){
    int avg = 0;

#if(1)
    for(j=3;j<nSamp+3;j++){
      avg += buf[j*NUM_CHANS+i];
    }
    avg /= nSamp;
#else
    avg = 0;
#endif


    for(j=0;j<3;j++){
      buf[j*NUM_CHANS+i] =0;
    }

    for(j=3;j<nSamp+3;j++){
      buf[j*NUM_CHANS+i] -= avg;
   }
  }

  for(i=3*NUM_CHANS;i<NUM_CHANS*(nSamp+3);i++) fprintf(baseline, "%d\n", (int) buf[i]);

  free(buf);

}

void getData(unsigned long base, short *baseline){

  int nSamp = nSamples(base, -1);

  /* user will need to delete this */
  short *buf = calloc(sizeof(buf[0]), nSamp*NUM_CHANS);

  reset(base);
  triggerSource(base, 0x2); /* rising edge? */
  /* triggerSource(base, 0x6); /* falling edge? */

  while(1){
    int i,chan;
    arm(base);
    waitAcq(base,1000);
    readRam(base,chan,baseline, &buf[0], &buf[nSamp], &buf[2*nSamp], &buf[3*nSamp]);
    for(chan=0;chan<NUM_CHANS;chan++){
      printf("wfd%d\t", chan+1);
      for(i=0; i < nSamp; i++) printf("%f\t", (buf[i+chan*nSamp]-2048)/4.096);
      printf("\n");
      }
  }
}

extern int init(long *BHandle);

#include <unistd.h>
int main(int argc, char ** argv) {

  int status = init(&BHandle);
  if(status != cvSuccess){
    fprintf(stderr,"\n\n Error %d opening the device\n", status);
    exit(1);
  }

  int nSamp;  
  fprintf(stderr, "nsamp = %d\n", nSamp=nSamples(0x4800000, -1));

  // if a filename was passed in, read it
  if(argc<2) {
    int i, temp;
    // else assume that we should take data (forever)
    // FIXME read in background data from stdin?
    
    short* background = calloc(sizeof(background[0]), (nSamp+3)*NUM_CHANS);
    for(i=0;i<nSamp*NUM_CHANS;i++) { temp=0; scanf("%d", &temp); background[i]=temp; }
    getData(0x4800000, background);
    free(background);

#if(0)
    fprintf(stderr, "%s\n", "invalid filename" );
    exit(1);
#endif
  }
  
  FILE *fp = fopen(argv[1], "w");
  if(!fp){
    fprintf(stderr, "Could not open file %s\n", argv[1]);
  }
 
  /* FIXME - should be a user arg... */
  takeBackground(0x4800000, fp);
  
  fclose(fp);
  
  exit(0);
}
