/*
Notice: This computer software was prepared by Brookhaven Science Associates, LLC and [individual author],
hereinafter the Contractor, under Contract DE-AC02-98CH10886 with the Department of Energy (DOE).
All rights in the computer software are reserved by DOE on behalf of the United States Government and the Contractor as
provided in the Contract. You are authorized to use this computer software for
Governmental purposes but it is not to be released or distributed to the public. NEITHER
THE GOVERNMENT NOR THE CONTRACTOR MAKES ANY WARRANTY,
EXPRESS OR IMPLIED, OR ASSUMES ANY LIABILITY FOR THE USE OF THIS
SOFTWARE. This notice including this sentence must appear on any copies of this
computer software.
(End of Notice)
*/
//
// File:   xmlParse.cc
// Author: Ben Pucci
//
// Created on November 14, 2007, 11:48 AM
//

#include <stdlib.h>
#include <stdio.h>
#include <strings.h> // strcasecmp()
#include <time.h> /* time() */
 
#include <stdarg.h>
#define LINUX
#include "CAENVMElib.h"
#undef LINUX

static const char *commentString = "!";
const unsigned long maxEventsPerSpill = 20000;

typedef struct {
  unsigned long numChannels; /* total # of channels */
  unsigned long activeChannels; /* no active channels means inactive module */
  unsigned long blts; /* # of bytes per "trigger" */
  unsigned long addr; /* VME address */
  CVAddressModifier am;
  CVDataWidth dtsize; /* width to read data into buffer */
  unsigned long *buff;
} MODULE;

extern "C" void unpackCaenScaler(FILE *outputF, MODULE *module, unsigned long i);
extern "C" void unpackCaenQDC(FILE *outputF, MODULE *module, unsigned long i);
extern "C" void unpackCaenTDC(FILE *outputF, MODULE *module, unsigned long i);
extern "C" void unpackCaenFADC(FILE *outputF, MODULE *module, unsigned long i);
extern "C" void unpackTimestamp(FILE *outputF, MODULE *module, unsigned long i){
  struct timespec *times = (struct timespec *) module->buff;
  printf("%u\t%u\t", (unsigned) times[i].tv_sec, (unsigned) times[i].tv_nsec);
  fprintf(outputF,"%u\t%u\t", (unsigned) times[i].tv_sec, (unsigned) times[i].tv_nsec);
  }
extern "C" void initUnpackFuncs(MODULE *module);

extern "C" {
  int runInterruptLoop(long BHandle, MODULE *modules, int irq, unsigned int *incVal, const volatile unsigned int *incAddr, unsigned long maxEventsPerSpill, time_t timeLimit);
  int init(long *BHandle);
}

static int text2size(const char *s){
  int size=0;
  if(!s)return 0;
  sscanf(s,"%i",&size);
  return size;
}
static CVAddressModifier text2am(const char *s){

  /* FIXME - should identify non-BLT devices (SIS and CAEN scalers) */
  /* Those should use cvA32_U_DATA */
  CVAddressModifier am = cvA32_U_BLT;
  if(strcasecmp(s,"A16")==0)am = cvA16_U;
  return am;
}
static CVDataWidth text2width(const char *s){
  CVDataWidth width = cvD32;
  if(strcasecmp(s,"D16")==0)width = cvD16;
  if(strcasecmp(s,"D8")==0)width = cvD8;
  return width;
}


#include <XmlDom/XDNode.hxx>
#include <XmlDom/XDDocument.hxx>

int main(int argc, char ** argv) {
    
    //the xml document
    XDDocument xddoc;
    
    int retval;
    int printDate = 0;

    // check for one argument in one position
    if(argc>1 && !strcmp(argv[1], "-date")) { printDate = ~0; argv++; argc--; }
    
    // if a filename was passed in, read it
    if(argc<2) {
        fprintf(stderr, "%s\n", "invalid filename" );
        exit(1);
    }

    retval = xddoc.read(argv[1]);
    if(retval) {
        fprintf(stderr, "%s\n", xddoc.getLastError() );
        exit(1);
        }
    /* get the root node */
    const XDNode *root = xddoc.getRootNode();
    
    /* first get the output data file, this is in the header node */
    const XDNode *header = root->getChild("header");
    
    /************* our output data file *************/
    const char *output = (header->getChild("output"))->getValue();
    
    fprintf(stderr, "Output File: %s\n", output);
    FILE *outputF = fopen(output, "a");
    if(!outputF){ fprintf(stderr, "Could not open file %s\n", output); exit(1); }
    
    /************* # of events to record *************/
    const char *nEventsS = (header->getChild("events"))->getValue();
    int nEvents = atoi(nEventsS);

    fprintf(stderr, "#Events: %d\n", nEvents);

    /************* time limit for experiment *************/
    time_t timeout  = -1; // forever
    if(header->getChild("timeout")) {
      const char *tLimit = (header->getChild("timeout"))->getValue();
      if(tLimit) timeout  = time(0)+atoi(tLimit);
      } 
    
    fprintf(stderr, "Time limit: %u\n", (unsigned) timeout);
    
    static MODULE modules[100]; // 4 VME crate's worth
    int moduleIndex = 0;

    /* the first module node in this document */
    const XDNode *module = (root->getChild("module"));
   
    /* scaler channel by default */
    static volatile const unsigned int *incrementAddr = (unsigned int *) (0x8384000 + 0x200 + 15*(sizeof(long)));
    unsigned int increment, incrChan=0 /* channel 0 by default*/, numChans;
    

    /* read in each individual module in the XML */
    while (module != NULL) {

      /* can we get the type? */
      const XDAttr *attr =  module->getAttr("type");
      const char *type = attr->getValue();

      fprintf(stderr, "module type = %s\n", type);

        /* make sure we have a module node */
        if (strcmp("module", module->getName()) == 0) {
            
            /* the address of this module */
            const char *address = (module->getChild("address"))->getValue();
            /* the address space of this module */
            const char *addrspace = (module->getChild("addrspace"))->getValue();
            /* the data width of this module */
            const char *datawidth = (module->getChild("datawidth"))->getValue();
            /* the data size for this module */
            const char *datasize = (module->getChild("datasize"))->getValue();
            
            /* now read the channels node */
            const XDNode *channels = module->getChild("channels");
            /* read how many total channels this module has !NOT JUST ACTIVE! */
            const char *chansize = (channels->getAttr("size"))->getValue();
	    numChans = text2size(chansize);
            /* the first single channel node contained within this channels node */
            const XDNode *channel = channels->getFirstChild();
           
            const char *storedModuleName = (module->getAttr("name"))->getValue();
            fprintf(stderr, "Module %s, Address: %s, AddrSpace: %s, DataWidth: %s, DataSize: %s, Channels: %s\n", 
                    storedModuleName, address, addrspace, datawidth, datasize, chansize);

	    unsigned long chanMask = 0;
            /* read all the channel nodes, only active channels will show up here */
            /* e.g. 1,2,5,6,7 out of 16 will show up */
            while (channel != NULL) {

                if(storedModuleName) { 
                    fprintf(outputF,"%s%s\n", commentString, (module->getAttr("name"))->getValue());
                    fprintf(stdout, "%s%s\n", commentString, (module->getAttr("name"))->getValue());
                    }
                storedModuleName=NULL;

                /* the name of the first active channel for output file header */
                const char *name = (channel->getAttr("name"))->getValue();
                /* the number of the active channel from [0-size) */
                const char *chanNum = channel->getValue();
                
                fprintf(stderr, "Channel, Name: %s, Number: %s\n", name, chanNum);
                fprintf(outputF,"%s%s\n", commentString, name);
                fprintf(stdout, "%s%s\n", commentString, name);

		/* accumulate active channels */
		/* keep track for inc (if needed) */
		incrChan = atoi(chanNum);
		chanMask |= 1<<incrChan;

                /* handle incrementer definition */

                /* get the next active channel or NULL if no more */
                channel = channel->getNextSibling();
            }
            
#if(0)
	    typedef struct {
	      int active; /* define active channels as well? */
	      unsigned long blts;
	      unsigned long addr;
	      CVAddressModifier am;
	      CVDataWidth dtsize;
	      unsigned long *buff;
	    } MODULE;
	    
#endif
            modules[moduleIndex].numChannels = numChans;
	    modules[moduleIndex].activeChannels = chanMask;

#if(0)
            modules[moduleIndex].activeChannels = ~0;
#endif
            modules[moduleIndex].addr = text2size(address);
	    modules[moduleIndex].blts = text2size(datasize);
	    modules[moduleIndex].am = text2am(addrspace);
	    modules[moduleIndex].dtsize = text2width(datasize);
	    modules[moduleIndex].buff = (unsigned long *) malloc(maxEventsPerSpill*modules[moduleIndex].blts*sizeof(modules[moduleIndex].buff));
            memset(modules[moduleIndex].buff, 0, maxEventsPerSpill*modules[moduleIndex].blts*sizeof(modules[moduleIndex].buff));

            /* handle incrementer definition */
	    if(strcmp(type, "incrementer")==0){
	      int chanDataSize = modules[moduleIndex].blts/numChans;
              incrementAddr = (unsigned int *) (modules[moduleIndex].addr + incrChan*chanDataSize);
	      fprintf(stderr, "incrementAddr = 0x%x, using channel # %d (last channel defined), data size %d\n", incrementAddr, incrChan, chanDataSize);
              }
	    
            /* handle trigger definition */
	    if(strcmp(type, "trigger")==0){
              fprintf(stderr, "Make sure this module is being triggered\n");
	      if(modules[moduleIndex].activeChannels==0)fprintf(stderr, "Warning - no channels defined on trigger module\n");
	    }
	    else {
	      fprintf(stderr, "Make sure this module is NOT being triggered\n");
            }

	    moduleIndex++;
        } // strcmp "module"
        
	
        /* get the next module node or NULL if there are none */
        module = module->getNextSibling();
    }

// ADD_TIMESTAMP_PSEUDO_MODULE
   modules[moduleIndex].numChannels = 1;
   modules[moduleIndex].activeChannels = 1;
   modules[moduleIndex].addr = 0;
   modules[moduleIndex].blts = 8;
   // modules[moduleIndex].am = 0;
   // modules[moduleIndex].dtsize = 4;
   modules[moduleIndex].buff = (unsigned long *) malloc(maxEventsPerSpill*modules[moduleIndex].blts); 

    long          BHandle;
    int           status;
    
#if(0)
    MODULE modules[] = {
      
      
      /* list of active modules */
      { 1, 80,   0x3020000, cvA32_U_BLT, cvD32},
      { 1, 136,  0x3040000, cvA32_U_BLT, cvD32},
      { 1, 72,   0x3080000, cvA32_U_BLT, cvD32},
      { 1, 72,   0x30e0000, cvA32_U_BLT, cvD32},
      { 1, 4096, 0x2800000, cvA32_U_BLT, cvD32},
      { 0 }
      
    };
    
#endif

    status = init(&BHandle);
    if( status != cvSuccess )
      {
	fprintf(stderr, "\n\n Error %d opening the device\n", status);
	exit(1);
      }


    initUnpackFuncs(modules);

    int events = 0;

    unsigned int oldValue;

    CAENVME_ReadCycle(BHandle,(unsigned long) incrementAddr, &increment,cvA32_U_DATA,cvD32);
    /* wait for incrementer to change one time before diving in */
    oldValue = increment;
    do {
         CAENVME_ReadCycle(BHandle,(unsigned long) incrementAddr, &increment,cvA32_U_DATA,cvD32);
    } while (increment == oldValue);

#if(0)
    /* debug code to try to understand why BLT of all SIS channels does not seem to
	work - may need BUS analyzer to know for sure - till then we will only use 
	SIS scaler for its incrementer role */

    char buf[1024]; int nb;

    CAENVME_BLTReadCycle(BHandle,(unsigned long)(0x8384000 + 0x200), buf, 64, cvA32_U_BLT,cvD32, &nb);
    CAENVME_BLTReadCycle(BHandle,(unsigned long)(0x8384000 + 0x200 + 15*(sizeof(long))), buf, 4, cvA32_U_BLT,cvD32, &nb);
#endif


    /* run one "throw away" cycle to get everything synced-up */
    /* give up after 30 seconds */
    (void) runInterruptLoop(BHandle, modules, 3, &increment, incrementAddr, 100, 30+time(0));
    /* FIXME -- if appropriate, do a second one to flush old data */
    /* (void) runInterruptLoop(BHandle, modules, 3, &increment, incrementAddr, 100, 30+time(0)); */



#define min(a,b) (a>b?b:a)

    for(;;){

      unsigned long eventsLeft = nEvents-events;

      int nTriggers = runInterruptLoop(BHandle, modules, 3, &increment, incrementAddr, min(maxEventsPerSpill,eventsLeft),timeout);

#if(1)
      fprintf(stderr, "#triggers = %d, incrementer value is now %d, at %d\n", nTriggers, increment, (int) time(0));
#endif

      if(nTriggers<0) break; // something went wrong 

#if(1)
      /* write one spill worth of data */
      for(int i=0;i<nTriggers;i++){
	for(size_t j=0;j<sizeof(modules)/sizeof(modules[0]);j++){
	  if(modules[j].activeChannels) {
	    
	    /* For now - let's infer how to unpack data based on data width */
	    /* Later we'll add it to the XML - maybe even use plug-ins? */
	    switch(modules[j].blts){
	      
	    case 64:
	      unpackCaenScaler(outputF, &modules[j], i);
	      break;
	    case 72:
            case 136: // dual-range
	      unpackCaenQDC(outputF, &modules[j], i);
	      break;
	    case 1024: // multi-hit
	      unpackCaenTDC(outputF, &modules[j], i);
	      break;
	    case 4096:
	      unpackCaenFADC(outputF, /* calData, */ &modules[j], i);
	      break;
            case 8:
              if(printDate)unpackTimestamp(outputF, &modules[j], i);
	    default:
	      break;
	    } /* switch */
	  } /* if active */
	} /* all modules */
	fprintf(outputF, "\n"); fflush(outputF);
        fprintf(stdout,  "\n"); fflush(stdout);
      } /* all triggers */
#endif
      
      events += nTriggers; 
      if(events >= nEvents)break;
    } /* until the entire data set is complete */
    
    fclose(outputF);
    
    return(0);
    
}
