/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
// package nsrlDataAcquisition;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.Vector;
import java.util.zip.GZIPOutputStream;
import javax.swing.BorderFactory;
import javax.swing.DefaultListSelectionModel;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SwingUtilities;
import javax.swing.table.AbstractTableModel;

/**
 * This is threaded data console. It mimics the original console output of the
 * data collected except now formats the data into columns. Each column is named
 * and associated with the corresponding channel of the currently running
 * configuration. It displays the last 5 full lines read from the end of the
 * data file.
 * 
 * @author Ben
 */
public class DataConsoleThreaded extends JFrame implements WindowListener {

    public DataConsoleThreaded(MainFrame mframeIn, String outputFile, Vector<String> colNames) {

        super("Data Console -- ");
        setPreferredSize(new Dimension(700, 400));
        setLayout(new GridBagLayout());
        GridBagConstraints c = new GridBagConstraints();
        addWindowListener(this);

        /* JTable to hold the channel values */
        _table = new JTable();
        _table.setAutoCreateColumnsFromModel(true);

        _columnNames = new String[colNames.size()+2];
        colNames.add("Time(sec)");
        colNames.add("Time(nanosec)");
        colNames.toArray(_columnNames);


        _rowData = new Vector<String[]>();

        _table.setModel(new DataTableModel());

        /* This header panel just shows the name of the output data file  */
        JPanel header = new JPanel();
        header.setBackground(new Color(99, 159, 198));
        header.setBorder(BorderFactory.createEmptyBorder(20, 10, 20, 10));
	_headerLabel = new JLabel("Data File: " + outputFile);
        header.add(_headerLabel);

        /* add the header panel */
        c.fill = GridBagConstraints.BOTH;
        c.anchor = GridBagConstraints.CENTER;
        c.gridx = 0;
        c.gridy = 0;
        c.weightx = .02;
        c.weighty = .02;

        add(header, c);

        /* add the JTable */
        c.gridx = 0;
        c.gridy = 1;
        c.weightx = .98;
        c.weighty = .98;
        c.anchor = GridBagConstraints.LINE_START;
        add(new JScrollPane(_table), c);

        setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);

        /* force table resize to even out all columns */
        _table.doLayout();

        /* create and start the thread to update the gui */
        _worker = new DataUpdateWorker(outputFile, 4, 5);
        _worker.start();
	pack();
	if(mframeIn!=null) setLocationRelativeTo(mframeIn);
	setVisible(true);
    }

    //Added so an open window can know a new acquistion has started, and update accordingly -MPM
    public void newAcquisition(String outputFile, Vector<String> colNames)
    {
	//stop the running update thread
        DataUpdateWorker worker = (DataConsoleThreaded.this.getWorker());
        if (worker != null) 
	{
            worker.cancel();
        }

	//setup new
        _columnNames = new String[colNames.size()+2];
        colNames.add("Time(sec)");
        colNames.add("Time(nanosec)");
        colNames.toArray(_columnNames);
	_rowData.clear();
	//_table.setModel(new DataTableModel());
	getModel().fireTableStructureChanged();
	//_table.repaint();
	_headerLabel.setText("Data File: " + outputFile);
	DataConsoleThreaded.this.doLayout();
        _worker = new DataUpdateWorker(outputFile, 4, 5);
        _worker.start();
    }

    private DataUpdateWorker getWorker() {
        return this._worker;
    }

    private Process getWriteProc() {
        return this._writeProc;
    }

    private Thread getRunner() {
        return this._procRunner;
    }

    /**
     * Custom TableModule class, limits the table to 50 values. Once 50 values
     * are reached it knocks the bottom (oldest) most values off the table. 
     */
    private class DataTableModel extends AbstractTableModel {

        public String getColumnName(int col) {
            return _columnNames[col].toString();
        }

        public int getRowCount() {
            return _rowData.size();
        }

        public int getColumnCount() {
            return _columnNames.length;
        }

        public Object getValueAt(int row, int col) {
            try {
                return _rowData.get(row)[col];
            } catch (Exception e) {
                System.out.println(e.getMessage());
            }
            return "";
        }

        public boolean isCellEditable(int row, int col) {
            return false;
        }

        public void setValueAt(Object value, int row, int col) {
            System.out.println("SET VALUE");
            _rowData.get(row)[col] = (String) value;
            fireTableCellUpdated(row, col);
        }

        public void addNewData(Vector<String[]> data) {

            if (_rowData.size() < 50) {
                for (int i = 0; i < data.size(); i++) {
                    _rowData.add(0, data.get(i));
                }

                fireTableRowsInserted(0, data.size());
            } else {
                int rowSize = _rowData.size();
                for (int i = 0; i < data.size(); i++) {
                    _rowData.add(0, data.get(i));
                    _rowData.remove(rowSize);
                }

                fireTableRowsInserted(0, data.size());
                fireTableRowsDeleted(((rowSize - data.size())), rowSize);

            }

            ((DefaultListSelectionModel) _table.getSelectionModel()).clearSelection();
            ((DefaultListSelectionModel) _table.getSelectionModel()).setSelectionInterval(0, data.size() - 1);
        }
        public static final long serialVersionUID = 1L;
    }

    public DataTableModel getModel() {
        return (DataTableModel) _table.getModel();
    }

    /**
     * If the windows is closed we need to stop any of the currently running
     * processes or threads.
     * 
     * @param e
     */
    public void windowClosed(WindowEvent e) {
        DataUpdateWorker worker = (DataConsoleThreaded.this.getWorker());
        if (worker != null) {
            worker.cancel();
        }
    }

    public void windowActivated(WindowEvent e) {
    }

    public void windowClosing(WindowEvent e) {
    }

    public void windowDeactivated(WindowEvent e) {
    }

    public void windowDeiconified(WindowEvent e) {
    }

    public void windowIconified(WindowEvent e) {
    }

    public void windowOpened(WindowEvent e) {
    }

    /**
     * Custom Thread to update the gui with the last 5 entries in the data file.
     */
    private class DataUpdateWorker extends Thread {

        public DataUpdateWorker(String filename, int wait, int max) {
            _lines = new CircleQueue(max);
            _waitTime = wait;
            _maxlines = max;
            _fileName = filename;
            _run = true;
        }

        public void run() {
            //used to check to see if the file has started being read.
            int status = 0;
            int cols = _columnNames.length;
            boolean newlines = false;

            //if the file doesn't exist this loops until it does or this thread
            //is cancelled.
            while (status == 0) {

                try {
                    BufferedReader br = new BufferedReader(new FileReader(_fileName));
                    status = 1;
                    String line = "";

                    /* Do until we are stopped */
                    while (_run) {

                        line = br.readLine();

                        /* Loop until we reach EOF */
                        while (line != null && _run) {

                            if (line != null) {
                                newlines = true;

                                /* Count how many tokens we have w/o StringTokenizer */
                                int charCount = 0;

                                if (!line.contains("!")) {
                                    for (int i = 0; i < line.length(); i++) {
                                        if (line.charAt(i) == 9) {
					    // if (line.charAt(i + 1) != 9) {
                                                charCount++;
						//} else if ((i + 2) == line.length()) {
						// charCount++;
						// }
                                        }
                                    }
                                }

                                /* If we have as many tokens as we have columns this
                                 * is a full line and we enqueue it.
                                 */
                                if (charCount >= cols) {
                                    _lines.enqueue(line);
				}
                            }

                            /* get the next line */
                            line = br.readLine();
                        }

                        /* only update the GUI if we have read new lines */
                        if (newlines) {
                            SwingUtilities.invokeLater(new UpdateGui(_lines));
                        }

                        /* clear the queue for the next read */
                        _lines.clear();
                        newlines = false;

                        /* wait x seconds before checking the file again */
                        sleep(_waitTime * 1000);
                    }
                } catch (FileNotFoundException fnf) {
                    try {
                        /* if there is no data file yet wait 5 seconds and try again */
                        if (_run) {
                            sleep(5000);
                        } /* if we are stopped make sure we exit the loop */ else {
                            status = 1;
                        }
                    } catch (InterruptedException ie) {
                    }
                } catch (IOException ioe) {
                } catch (InterruptedException ire) {
                }
            }
        }

        public void cancel() {
            this._run = false;
        }
        private CircleQueue _lines;
        private int _waitTime;
        private int _maxlines;
        private String _fileName;
        private boolean _header;
        private boolean _run;
    }

    /* A runnable used to process the data chunks and update the gui */
    private class UpdateGui implements Runnable {

        private String[] update;
        private Vector<String[]> parsed;

        public UpdateGui(CircleQueue lines) {

            //create objects and then get the oldest position
            update = new String[lines.getSize()];
            parsed = new Vector<String[]>(lines.getSize());
            int position = (lines.getPosition() + 1) % lines.getSize();

            /* fill our array with newest --> oldest records then wait to run */
            int i = update.length - 1;
            while (i >= 0) {
                if (lines.get(position) != null) {
                    update[i] = new String(lines.get(position));
                } else {
                    update[i] = "";
                }

                position = (position + 1) % update.length;
                i--;
            }
        }

        /* use the regex expression to split out the data */
        public void run() {

            for (int i = (update.length - 1); i >= 0; i--) {
                parsed.add(update[i].split("\\s+"));
            }

            /* update the gui */
            DataConsoleThreaded.this.getModel().addNewData(parsed);
            /* clear the old data */
            parsed.removeAllElements();
        }
    }

    /* Circular Queue written to ease java vector overhead for element shifting */
    private class CircleQueue {

        private String[] _queue;
        private int _pos;

        public CircleQueue(int size) {
            _queue = new String[size];
            _pos = -1;
        }

        public void enqueue(String data) {
            _pos = (_pos + 1) % _queue.length;
            _queue[_pos] = data;
        }

        public int getPosition() {
            return this._pos;
        }

        public int getSize() {
            return this._queue.length;
        }

        public void clear() {
            for (int i = 0; i < _queue.length; i++) {
                _queue[i] = null;
            }
            this._pos = -1;
        }

        public String get(int i) {
            if (i < _queue.length) {
                return _queue[i];
            } else {
                return null;
            }
        }
    }
    public static final long serialVersionUID = 1L;
    private JTable _table;
    private String[] _columnNames;
    private Vector<String[]> _rowData;
    private DataUpdateWorker _worker;
    private Process _writeProc;
    private Thread _procRunner;
    private boolean window;
    private JLabel _headerLabel;
}
